#include        "gpib.h"
#include        "formatio.h"
#include        "tsxp.h"

/* Thurlby-Thandar TSXP series power supplies instrument driver            */
/* LabWindows 2.0 Multi-Instrument Driver                                  */
/* Original Release: Jan 1992                                              */
/* By: John Tothill, Thurlby Thandar Instruments Ltd.                      */
/* Originally written in C                                                 */
/* Modification History: V2.0 updated to latest standard and               */
/*                       and multi-instrument Apr 1995                     */
/*=========================================================================*/

/*= INSTRUMENT TABLE ======================================================*/
/* address array: contains the GPIB addresses of opened instruments.       */
/* bd array: contains the device descriptors returned by OpenDev.          */
/* instr_cnt: contains the number of instruments open of this model type.  */
/* tsxp_err: the error variable for the instrument module                */
/*=========================================================================*/
static int address[tsxp_MAX_INSTR + 1];
static int bd[tsxp_MAX_INSTR + 1];
static int instr_cnt;
/*static int tsxp_err;*/
/*= INTERNAL DATA =========================================================*/
/* 'buffer' is a buffer for GPIB I/O strings */
static char buffer[100];
static long tsxp_model;
static int tsxp_err;

/*= UTILITY ROUTINES ======================================================*/
int tsxp_open_instr (int);
int tsxp_close_instr (int);
int tsxp_invalid_integer_range (int, int, int, int);
int tsxp_device_closed (int);
int tsxp_read_data (int, char *, int);
int tsxp_write_data (int, char *, int);
/*=========================================================================*/
/* Function: Initialize                                                    */
/* Purpose:  This function opens the instrument                            */
/*=========================================================================*/
int tsxp_init (addr, instrID)
int addr;
int *instrID;
{
int  ID;
int ident;
    ident = 0;
    if (tsxp_invalid_integer_range (addr, 0, 30, -1) != 0)
        return tsxp_err;

    tsxp_err = 0;
    ID = tsxp_open_instr (addr);
    if (ID <= 0)
        return tsxp_err;

    ibtmo (bd[ID], 14);   /* 30 sec timeout */
    if (ibsta & 0x8000)
        {
        tsxp_err = 300;
        return tsxp_err;
        }
    ibeot (bd[ID], 1);    /* enable end message */
    if (ibsta & 0x8000)
        {
        tsxp_err = 300;
        return tsxp_err;
        }

    ibclr (bd[ID]);
    if (tsxp_write_data(ID,"*CLS\012",5))  /* clear status */
        return tsxp_err;
    if (tsxp_write_data(ID,"*IDN?\012",5))  /* get identifier string */
        return tsxp_err;
    if (tsxp_read_data(ID,buffer,50))
        return tsxp_err;
    Scan(buffer,"THURLBY-THANDAR,TSX%d,0,0,00",&ident); /* check idn */
    if (ident == 3510 || ident == 1820)
        tsxp_model = ident;
    else     /* error unknown ident */
        {
        tsxp_err = 223;
        return tsxp_err;
        }

    *instrID = ID;
    return tsxp_err;
}
/*=========================================================================*/
/* Function: Set output                                                    */
/* Purpose:  This function sets Voltage and Current                        */
/*=========================================================================*/
int tsxp_set_op(instrID, voltage,current)
int instrID;
double voltage;
double current;
{
    if (tsxp_invalid_integer_range (instrID, 1, tsxp_MAX_INSTR, -1) != 0)
        return tsxp_err;
    if (tsxp_device_closed (instrID))
        return tsxp_err;

    Fmt(buffer,"V %f;I %f\012",voltage,current);
    if (tsxp_write_data(instrID, buffer,NumFmtdBytes()))
                return tsxp_err;

    return tsxp_err;
}
/*=========================================================================*/
/* Function: Set OVP                                                       */
/* Purpose:  This function sets the OVP value                              */
/*=========================================================================*/
void tsxp_set_ovp(instrID, ovp)
int instrID;
double ovp;
{
    if (tsxp_invalid_integer_range (instrID, 1, tsxp_MAX_INSTR, -1) != 0)
        return;
    if (tsxp_device_closed (instrID))
        return;

    Fmt(buffer,"OVP %f\012",ovp);
    if (tsxp_write_data(instrID, buffer,NumFmtdBytes()))
        return;
    tsxp_err = 0;
}
/*=========================================================================*/
/* Function: Set damping                                                   */
/* Purpose:  This function sets the damping of the current meter ON or OFF */
/*=========================================================================*/
void tsxp_set_damping (instrID, dmpg)
int instrID;
int dmpg;
{
    if (tsxp_invalid_integer_range (instrID, 1, tsxp_MAX_INSTR, -1) != 0)
        return;
    if (tsxp_invalid_integer_range (dmpg, 0, 1, -1) != 0)
        return;
    if (tsxp_device_closed (instrID))
        return;

    Fmt(buffer,"DAMPING %d\012",dmpg);
    if (tsxp_write_data(instrID, buffer,NumFmtdBytes()))
        return;
    tsxp_err = 0;  /* success */
}
/*=========================================================================*/
/* Function: Set output switch                                             */
/* Purpose:  This function sets the output switch status ON or OFF         */
/*=========================================================================*/
int tsxp_op_switch (instrID, op_stat)
int instrID;
int op_stat;
{
    if (tsxp_invalid_integer_range (instrID, 1, tsxp_MAX_INSTR, -1) != 0)
        return tsxp_err;
    if (tsxp_invalid_integer_range (op_stat, 0, 1, -2) != 0)
        return tsxp_err;
    if (tsxp_device_closed (instrID))
        return tsxp_err;

    Fmt(buffer,"OP %d\012",op_stat);
    if (tsxp_write_data(instrID, buffer,NumFmtdBytes()))
        return tsxp_err;
    return tsxp_err;  /* success */
}
/*=========================================================================*/
/* Function: Set status                                                    */
/* Purpose:  This function sets the status reporting capabilities of the   */
/*           instrument as specifird by IEEE 488.2                         */
/*=========================================================================*/
int tsxp_set_status (instrID, reg_type,byte_value)
int instrID;
int reg_type;
int byte_value;
{
char str[5];
    if (tsxp_invalid_integer_range (instrID, 1, tsxp_MAX_INSTR, -1) != 0)
        return tsxp_err;
    if (tsxp_invalid_integer_range (reg_type, 0, 3, -2) != 0)
        return tsxp_err;
    if (tsxp_invalid_integer_range (byte_value, 0, 255, -3) != 0)
        return tsxp_err;
    if (tsxp_device_closed (instrID))
        return tsxp_err;

    switch (reg_type) {
    case 0:
        Fmt(str,"*SRE");
        break;
    case 1:
        Fmt(str,"*ESE");
        break;
    case 2:
        Fmt(str,"LSE");
        break;
    case 3:
        Fmt(str,"*PRE");
        break;
    }
    Fmt(buffer,"%s %d\012",str,byte_value);
    if (tsxp_write_data(instrID, buffer,NumFmtdBytes()))
        return tsxp_err;
    return tsxp_err;  /* success */
}
/*=========================================================================*/
/* Function: Read Volts                                                    */
/* Purpose:  This function allows the output voltage of the psu to be read */
/*=========================================================================*/
void tsxp_read_volts (instrID, voltage)
int instrID;
double *voltage;
{
    if (tsxp_invalid_integer_range (instrID, 1, tsxp_MAX_INSTR, -1) != 0)
        return;
    if (tsxp_device_closed (instrID))
        return;

    Fmt(buffer,"VO?\012");  /* read voltage */
    if (tsxp_write_data(instrID, buffer,NumFmtdBytes()))
        return;
    if (tsxp_read_data(instrID, buffer,50))
        return;
    Scan(buffer,"%fV",voltage); /* extract voltage value */
    return;  /* success */
}
/*=========================================================================*/
/* Function: Read Current                                                  */
/* Purpose:  This function allows the output current of the psu to be read */
/*=========================================================================*/
void tsxp_read_current(instrID, current)
int instrID;
double *current;
{
    if (tsxp_invalid_integer_range (instrID, 1, tsxp_MAX_INSTR, -1) != 0)
        return;
    if (tsxp_device_closed (instrID))
        return;

    Fmt(buffer,"IO?\012");  /* read current */
    if (tsxp_write_data(instrID, buffer,NumFmtdBytes()))
        return;
    if (tsxp_read_data(instrID, buffer,50))
        return;
    Scan(buffer,"%fA",current); /* extract current value */
    return;  /* success */
}
/*=========================================================================*/
/* Function: Read status                                                   */
/* Purpose:  This function allows the value to be read from the selected   */
/*           status register                                               */
/*=========================================================================*/
int tsxp_read_status (instrID, reg_type, byte)
int instrID;
int reg_type;
int *byte;
{
char str[7];
    if (tsxp_invalid_integer_range (instrID, 1, tsxp_MAX_INSTR, -1) != 0)
        return tsxp_err;
    if (tsxp_invalid_integer_range (reg_type, 0, 6, -2) != 0)
        return tsxp_err;
    if (tsxp_device_closed (instrID))
        return tsxp_err;

    switch (reg_type) {
    case 0:
        Fmt(str,"*STB?\012");
        break;
    case 1:
        Fmt(str,"*ESR?\012");
        break;
    case 2:
        Fmt(str,"LSR?\012");
        break;
    case 3:
        Fmt(str,"*SRE?\012");
        break;
    case 4:
        Fmt(str,"*ESE?\012");
        break;
    case 5:
        Fmt(str,"LSE?\012");
        break;
    case 6:
        Fmt(str,"*PRE?\012");
        break;
    }
    if (tsxp_write_data(instrID, str,NumFmtdBytes()))
        return(tsxp_err);
    if (tsxp_read_data(instrID, buffer,50))
        return(tsxp_err);
    Scan(buffer,"%d",byte); /* extract value */
    return tsxp_err;  /* success */
}
/*=========================================================================*/
/* Function: Store                                                         */
/* Purpose:  This function saves the complete state of the psu to local    */
/*           non-volatile memory                                           */
/*=========================================================================*/
int tsxp_store (instrID, store_no)
int instrID;
int store_no;
{
    if (tsxp_invalid_integer_range (instrID, 1, tsxp_MAX_INSTR, -1) != 0)
        return tsxp_err;
    if (tsxp_invalid_integer_range (store_no, 1, 25, -2) != 0)
        return tsxp_err;
    if (tsxp_device_closed (instrID))
        return tsxp_err;

    Fmt(buffer,"*SAV %d\012",store_no);
    if (tsxp_write_data(instrID, buffer,NumFmtdBytes()))
        return tsxp_err;
    return tsxp_err;  /* success */
}

/*=========================================================================*/
/* Function: Recall                                                        */
/* Purpose:  This function recalls the psu setup from local non-volatile   */
/*           memory                                                        */
/*=========================================================================*/
int tsxp_recall (instrID, store_no)
int instrID;
int store_no;
{
    if (tsxp_invalid_integer_range (instrID, 1, tsxp_MAX_INSTR, -1) != 0)
        return tsxp_err;
    if (tsxp_invalid_integer_range (store_no, 1, 25, -2) != 0)
        return tsxp_err;
    if (tsxp_device_closed (instrID))
        return tsxp_err;

    Fmt(buffer,"*RCL %d\012",store_no);
    if (tsxp_write_data(instrID, buffer,NumFmtdBytes()))
        return tsxp_err;
    return tsxp_err;  /* success */
}
/*=========================================================================*/
/* Function: Send command                                                  */
/* Purpose:  This function allows any command to be sent to the psu        */
/*=========================================================================*/
int tsxp_send_cmd (instrID, cmd_buf)
int instrID;
char *cmd_buf;
{
char output_buf[2000];
    if (tsxp_invalid_integer_range (instrID, 1, tsxp_MAX_INSTR, -1) != 0)
        return tsxp_err;
    if (tsxp_device_closed (instrID))
        return tsxp_err;

    Fmt(output_buf,"%s",cmd_buf);
    if (tsxp_write_data(instrID, cmd_buf,NumFmtdBytes()))
        return tsxp_err;
    return tsxp_err;  /* success */
}
/*=========================================================================*/
/* Function: Get response                                                  */
/* Purpose:  This function allows a response to be collected from a        */
/*           previously executed send command                              */
/*=========================================================================*/
int tsxp_get_response (instrID, cmd_buf, bcount)
int instrID;
char *cmd_buf;
int *bcount;
{
    if (tsxp_invalid_integer_range (instrID, 1, tsxp_MAX_INSTR, -1) != 0)
        return tsxp_err;
    if (tsxp_device_closed (instrID))
        return tsxp_err;

    if (tsxp_read_data(instrID, cmd_buf,2000))
        return tsxp_err;
    *bcount = ibcnt;
    return tsxp_err;
}
/*=========================================================================*/
/* Function: Send from file                                                */
/* Purpose:  This function sends data from a file to the psu               */
/*=========================================================================*/
int tsxp_send_from_file (instrID, fname)
int instrID;
char *fname;
{
    if (tsxp_invalid_integer_range (instrID, 1, tsxp_MAX_INSTR, -1) != 0)
        return tsxp_err;
    if (tsxp_device_closed (instrID))
        return tsxp_err;

    ibwrtf (bd[instrID], fname);
    if (ibsta & 0x8000) /* any errors */
        tsxp_err = 300;
    else
        tsxp_err = 0;  /* success */
    return tsxp_err;
}
/*=========================================================================*/
/* Function: Get to file                                                   */
/* Purpose:  This function gets data from the psu to a file                */
/*=========================================================================*/
int tsxp_get_to_file (instrID, fname)
int instrID;
char *fname;
{
    if (tsxp_invalid_integer_range (instrID, 1, tsxp_MAX_INSTR, -1) != 0)
        return tsxp_err;
    if (tsxp_device_closed (instrID))
        return tsxp_err;

    ibrdf (bd[instrID], fname);
    if (ibsta & 0x8000) /* any errors */
        tsxp_err = 300;
    else
        tsxp_err = 0;  /* success */
    return tsxp_err;
}
/*=========================================================================*/
/* Function: Close                                                         */
/* Purpose:  This function closes the instrument.                          */
/*=========================================================================*/
int tsxp_close (instrID)
int instrID;
{
    if (tsxp_invalid_integer_range (instrID, 1, tsxp_MAX_INSTR, -1) != 0)
        return tsxp_err;

    if (tsxp_device_closed (instrID))
        return tsxp_err;

    tsxp_close_instr (instrID);
    return tsxp_err;
}
/*= UTILITY ROUTINES ======================================================*/

/*=========================================================================*/
/* Function: Open Instrument                                               */
/* Purpose:  This function locates and initializes an entry in the         */
/*           Instrument Table and the GPIB device table for the            */
/*           instrument.  If successful, the instrument ID is returned,    */
/*           else a -1 is returned.  The size of the Instrument Table can  */
/*           be changed in the include file by altering the constant       */
/*           tsxp_MAX_INSTR.                                               */
/*=========================================================================*/
int tsxp_open_instr (addr)
int addr;
{
    int i, instrID;

    instrID = 0;

/* Check to see if the instrument is already in the Instrument Table. */

    for (i=1; i<= tsxp_MAX_INSTR; i++)
        if (address[i] == addr)
            {
            instrID = i;
            i = tsxp_MAX_INSTR;
            }

/* If it is not in the instrument table, open an entry for the instrument. */

    if (instrID <= 0)
        for (i=1; i<= tsxp_MAX_INSTR; i++)
            if (address[i] == 0)
                {
                instrID = i;
                i = tsxp_MAX_INSTR;
                }

/* If an entry could not be opened in the Instrument Table, return an error.*/

    if (instrID <= 0)
        {
        tsxp_err = 220;
        return -1;
        }

/*  If the device has not been opened in the GPIB device table (bd[ID] = 0),*/
/*  then open it.                                                           */

    if (bd[instrID] <= 0)
        {
        if (instr_cnt <= 0)
            CloseInstrDevs("TPLP");
        bd[instrID] = OpenDev ("", "TPLP");
        if (bd[instrID] <= 0)
            {
            tsxp_err = 220;
            return -1;
            }
        instr_cnt += 1;
        address[instrID] = addr;
        }

/*  Change the primary address of the device    */

    if (ibpad (bd[instrID], addr) < 0)
        {
        tsxp_err = 233;
        return -1;
        }

    return instrID;
}
/*=========================================================================*/
/* Function: Close Instrument                                              */
/* Purpose:  This function closes the instrument by removing it from the   */
/*           GPIB device table and setting the address and the bd to zero  */
/*           in the Instrument Table.  The return value is equal to the    */
/*           global error variable.                                        */
/*=========================================================================*/
int tsxp_close_instr (instrID)
int instrID;
{
    if (bd[instrID] != 0)
        {
        CloseDev (bd[instrID]);
        bd[instrID] = 0;
        address[instrID] = 0;
        instr_cnt -= 1;
        }
    else
        tsxp_err = 221;

    return tsxp_err;
}
/*=========================================================================*/
/* Function: Invalid Integer Range                                         */
/* Purpose:  This function checks an integer to see if it lies between a   */
/*           minimum and maximum value.  If the value is out of range, set */
/*           the global error variable to the value err_code.  If the      */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int tsxp_invalid_integer_range (val, min, max, err_code)
int val;
int min;
int max;
int err_code;
{
  if ((val < min) || (val > max))
    {
    tsxp_err = err_code;
    return -1;
    }
  return 0;
}
/*=========================================================================*/
/* Function: Device Closed                                                 */
/* Purpose:  This function checks to see if the module has been            */
/*           initialized.  If the device has not been opened, a 1 is       */
/*           returned, 0 otherwise.                                        */
/*=========================================================================*/
int tsxp_device_closed (instrID)
int instrID;
{
    if (bd[instrID] <= 0)
        {
        tsxp_err = 232;
        return -1;
        }
    return 0;
}
/*=========================================================================*/
/* Function: Read Data                                                     */
/* Purpose:  This function reads a buffer of data from the instrument. The */
/*           return value is equal to the global error variable.           */
/*=========================================================================*/
int tsxp_read_data (instrID, buf, cnt)
int instrID;
char *buf;
int cnt;
{
    if (ibrd(bd[instrID], buf, (long)cnt) <= 0)
        tsxp_err = 231;
    else
        tsxp_err = 0;

    return tsxp_err;
}
/*=========================================================================*/
/* Function: Write Data                                                    */
/* Purpose:  This function writes a buffer of data to the instrument. The  */
/*           return value is equal to the global error variable.           */
/*=========================================================================*/
int tsxp_write_data (instrID, buf, cnt)
int instrID;
char *buf;
int cnt;
{
    if (ibwrt(bd[instrID], buf, (long)cnt) <= 0)
        tsxp_err = 230;
    else
        tsxp_err = 0;

    return tsxp_err;
}
/*= THE END ===============================================================*/
